{******************************************************************************}
{                                                                              }
{                           The Adjustable ComboBox                            }
{                                      By                                      }
{                                Robert Wittig                                 }
{                                                                              }
{ Description                                                                  }
{ -----------                                                                  }
{ This unit provides a ComboBox that allows sizing of the drop-down list box.  }
{ For simplicity, I have only offered two options:                             }
{        ddwParent : The drop-down list box is sized to the same width as the  }
{                    parent.                                                   }
{        ddwWidest : The drop-down list box is sized to be wide enough to      }
{                    allow for the width of the longest item + 10 pixels.      }
{ Regardless of which case is chosen above, the height of the list box is      }
{ determined by the DropDownCount property and the height of the first item.   }
{ The height of the first item is determined by ItemHeight, except for a       }
{ csOwnerDrawVariable ComboBox, where you may override the default ItemHeight  }
{ in the OnMeasureItem event.                                                  }
{                                                                              }
{ Also, when dropping down, the ComboBox checks to see if the drop-down will   }
{ extend, either left, right, bottom or top, past the edge of the screen.      }
{ If so, it attempts to adjust the position of the drop-down to get rid of     }
{ the problem.                                                                 }
{                                                                              }
{ Note that the standard ComboBox provided by Delphi does _not_ size the       }
{ drop-down list box correctly for csOwnerDrawVariable/csOwnerDrawFixed list   }
{ boxes.  Instead, it _always_ sizes the drop-down according to the current    }
{ font size, regardless of actual ItemHeight.                                  }
{                                                                              }
{ Installation/Use                                                             }
{ ----------------                                                             }
{ Install this control in the normal manner, using Options|Install Components. }
{                                                                              }
{ About the author                                                             }
{ ----------------                                                             }
{ Robert Wittig.  You are most likely to reach me at CompuServe account        }
{ 74601,242.  You can either send me e-mail or leave a message in the Borland  }
{ Delphi forum.                                                                }
{                                                                              }
{ Restrictions/Permissions/Warnings                                            }
{ ---------------------------------                                            }
{ The following restrictions apply to the use of this code.  You may use it    }
{ all you want with no royalties or run-time fees.  If you distribute it as    }
{ part of a larger package, and distribute source code, you must leave this    }
{ header in place, with possible additions by yourself.  You may not charge    }
{ for use or copies of this code beyond a nominal fee for compilation and      }
{ distribution.  In other words, you don't have the right to get money just    }
{ for reselling this code.  However, I have no problem with inclusion of this  }
{ code in a larger application.                                                }
{                                                                              }
{ This code is provided as-is.  If you encounter difficulties in using this    }
{ control, please e-mail me at one of the above addresses.  Time permitting,   }
{ I will attempt to address your problems, but I make no guarantees or         }
{ statments of liability with regard to this code or its use.                  }
{                                                                              }
{ Final Notes                                                                  }
{ -----------                                                                  }
{ With the messy stuff out of the way, I hope you'll like and enjoy using      }
{ this control.  This is tested only for Delphi 1.0, Windows 3.1.  I would     }
{ appreciate feedback as to its usefulness in other environments.              }
{                                                                              }
{ History   Author  Change                                                     }
{ --------  ------  ------                                                     }
{ 2/19/96   RCW     Written, debugged, commented by Robert Wittig (RCW)        }
{ 6/13/96   RCW     Moved sizing/changing code from CN_COMMAND to              }
{                   WM_WINDOWPOSCHANGING to achieve greater control over       }
{                   position of the drop-down.                                 }
{ 7/1/96    RCW     Fixed bug which caused GPF when removing AdjustComboBox    }
{                   from Form while in IDE.  Added code to preserve the current}
{                   selection when the combo box is re-created, for example    }
{                   when changing the style at runtime.  Finally, I've added   }
{                   the Reset procedure.  This is simply a masked call to      }
{                   RecreateWnd, giving you a chance to make changes to the    }
{                   ItemHeight or height of the edit-portion immediately       }
{                   visible.  For example, in a csOwnerDrawVariable            }
{                   AdjustComboBox, you can make a change to a global variable }
{                   or to ItemHeight, then call Reset to adjust the size of the}
{                   drop-down or the edit portion of the combo box.            }
{ 7/2/96    RCW     Overrided the DrawItem procedure to provide proper handling}
{                   of the DrawItem event when the ComboBox uses an owner-draw }
{                   style.  Delphi's default handling of the drawing will      }
{                   overwrite the button-portion of the ComboBox if the text is}
{                   longer than the edit-portion of the box.                   }
{                                                                              }
{******************************************************************************}
unit AdjstCmb;

interface
Uses Messages, WinProcs, WinTypes,
     Classes, Controls, Graphics, Menus, StdCtrls, SysUtils;

Type
    TDropDownWidth = ( ddwParent, ddwWidest );

    TCustomAdjustComboBox = Class ( TCustomComboBox )
    Private
           FComboListHandle : HWnd;
           FDropDownWidth   : TDropDownWidth;
           FSaveItemIndex   : Integer;
           FShowFocus       : Boolean;

           FDropDownInstance : Pointer;
           FOldWndProc       : Pointer;

           Procedure CNDrawItem ( Var Message : TWMDrawItem ); message CN_DRAWITEM;
    Protected
             Procedure CreateWnd; override;
             Procedure DestroyWnd; override;
             Procedure DrawItem ( Index : Integer; Rect : TRect; State : TOwnerDrawState ); override;
             Procedure WndProc   ( Var Message : TMessage ); override;

             Procedure DropDownWndProc ( Var Message : TMessage );
             Function  MaxWidth : Integer; virtual;
             Procedure RepositionDropDown ( Var X, Y : Integer; cX, cY : Integer ); virtual;
             Procedure ResizeDropDown ( Var cX, cY : Integer ); virtual;

             { Properties:
                 ComboListHandle  The window handle of the drop-down listbox.
                 DropDownWidth    Controls width of the drop-down.  A value of
                                  ddwParent provides the default behavior, while
                                  a value of ddwWidest expands the width of the
                                  drop-down so that the widest item in the list
                                  will be fully visible.
                 ShowFocus        For either of the owner-draw styles, this
                                  property controls whether or not the dashed
                                  focus box is displayed.  For non-owner-draw
                                  styles, this property has no effect. }
             Property ComboListHandle  : HWnd           Read FComboListHandle;
             Property DropDownWidth    : TDropDownWidth Read FDropDownWidth Write FDropDownWidth;
             Property ShowFocus        : Boolean        Read FShowFocus Write FShowFocus default True;
    Public
          Constructor Create ( aOwner : TComponent ); override;
          Destructor Destroy; override;

          { Reset is a thinly disguised call to RecreateWnd.  It can be used
            when you want to change the height of items in the drop-down (for
            both owner-draw styles) or the height of the edit-portion only (for
            the csOwnerDrawVariable style).  For the csOwnerDrawFixed style,
            call Reset after changing the ItemHeight property.  For the
            csOwnerDrawVariable style, you need to provide an OnMeasureItem
            event handler.  Then, call Reset and return different values when
            item height is requested. }
          Procedure  Reset;
    End;

    TAdjustComboBox = Class ( TCustomAdjustComboBox )
    Published
             Property Style; {Must be published before Items}
             Property Color;
             Property Ctl3D;
             Property DragMode;
             Property DragCursor;
             Property DropDownCount;
             Property DropDownWidth;
             Property Enabled;
             Property Font;
             Property ItemHeight;
             Property Items;
             Property MaxLength;
             Property ParentColor;
             Property ParentCtl3D;
             Property ParentFont;
             Property ParentShowHint;
             Property PopupMenu;
             Property ShowFocus;
             Property ShowHint;
             Property Sorted;
             Property TabOrder;
             Property TabStop;
             Property Text;
             Property Visible;

             Property OnChange;
             Property OnClick;
             Property OnDblClick;
             Property OnDragDrop;
             Property OnDragOver;
             Property OnDrawItem;
             Property OnDropDown;
             Property OnEndDrag;
             Property OnEnter;
             Property OnExit;
             Property OnKeyDown;
             Property OnKeyPress;
             Property OnKeyUp;
             Property OnMeasureItem;
    End;


Procedure Register;

implementation
Uses Forms;

Procedure Register;
Begin
     RegisterComponents ( 'Custom', [TAdjustComboBox]);
End;

Constructor TCustomAdjustComboBox.Create ( aOwner : TComponent );
Begin
     Inherited Create ( aOwner );

     { Create an ObjectInstance of the DropDownWndProc for later use
       in subclassing the drop-down list box }
     FDropDownInstance := MakeObjectInstance ( DropDownWndProc );

     FShowFocus := True;
End;

Destructor TCustomAdjustComboBox.Destroy;
Begin
     Inherited Destroy;

     { Free the earlier-created ObjectInstance of the DropDownWndProc }
     FreeObjectInstance ( FDropDownInstance );
End;


Procedure TCustomAdjustComboBox.CNDrawItem ( Var Message : TWMDrawItem );
Begin
     {*************** RCW 7/2/96 *******************************}
     { This little change allows you to decide if you want to   }
     { display the focus rectangle in the box or not.  To       }
     { disable the focus rectangle, set the ShowFocus to False. }
     {**********************************************************}
     With Message.DrawItemStruct^ Do
     Begin
          If ( ItemState And ODS_FOCUS <> 0 ) And Not FShowFocus
             Then ItemState := ItemState And Not ODS_FOCUS;
     End;

     Inherited;
End;

Procedure TCustomAdjustComboBox.CreateWnd;
Begin
     Inherited CreateWnd;

     {**************************** Rcw 7/2/96 *******}
     { Restore the ItemIndex saved when the ComboBox }
     { window was destroyed.                         }
     {***********************************************}
     ItemIndex := FSaveItemIndex;
End;

Procedure TCustomAdjustComboBox.DestroyWnd;
Begin
     {**************************** Rcw 7/2/96 *********}
     { Save the ItemIndex, so that it can be           }
     { restored the next time the ComboBox is created. }
     {*************************************************}
     FSaveItemIndex := ItemIndex;

     Inherited DestroyWnd;
End;

Procedure TCustomAdjustComboBox.DrawItem ( Index : Integer; Rect : TRect; State : TOwnerDrawState );
Var
   DrawItemEvent : TDrawItemEvent;
Begin
     {*************************** RCW 7/2/96 ***********************}
     { Here I override the default handling of the DrawItem event.  }
     { This is because, by default, the ComboBox control will       }
     { overwrite the button with text when the style is owner-draw. }
     { Using TextRect instead of TextOut (as is done in the VCL)    }
     { prevents this problem easily.                                }
     {**************************************************************}

     DrawItemEvent := OnDrawItem;
     If Assigned ( DrawItemEvent )
        Then DrawItemEvent ( Self, Index, Rect, State )
        Else Begin
             Canvas.FillRect ( Rect );
             Canvas.TextRect ( Rect, Rect.Left + 2, Rect.Top, Items[Index] );
        End;
End;

Procedure TCustomAdjustComboBox.WndProc ( Var Message : TMessage );
Var
   ClassName : Array[0..80] Of Char;
Begin
     {*************************** RCW 7/2/96 ***********************}
     { To manipulate the drop-down box, I need its window handle.   }
     { The easiest place (that I've found, at least) to retrieve    }
     { this handle is when the drop-down sends a WM_PARENTNOTIFY    }
     { message to its parent during creation.  When I see a         }
     { ComboLBox child control being created, I store its handle    }
     { and subclass its window procedure.                           }
     {**************************************************************}

     If Message.Msg = WM_PARENTNOTIFY
        Then With TWMParentNotify ( Message ) Do
             If ( Event = WM_CREATE ) And
                ( ChildWnd <> 0 )
                Then Begin
                     GetClassName ( ChildWnd, ClassName, 80 );
                     If StrIComp ( 'COMBOLBOX', ClassName ) = 0
                        Then Begin
                             { Store drop-down window handle. }
                             FComboListHandle := ChildWnd;

                             { Save old WndProc and substitute new WndProc. }
                             FOldWndProc := Pointer ( GetWindowLong ( FComboListHandle, GWL_WNDPROC ) );
                             SetWindowLong ( FComboListHandle, GWL_WNDPROC, LongInt ( FDropDownInstance ) );
                        End
                End;

     Inherited WndProc ( Message );
End;


Procedure TCustomAdjustComboBox.DropDownWndProc ( Var Message : TMessage );
Const
     Last : TPoint = ( X : 0; Y : 0 );
Begin
     {*********************** RCW 7/2/96 ***************************}
     { When the drop-down receives a WM_WINDOWPOSCHANGING message,  }
     { the ComboLBox has already decided where to position itself.  }
     { This makes the perfect place to change that decision.        }
     {**************************************************************}

     If Message.Msg = WM_WINDOWPOSCHANGING
        Then With TWMWindowPosMsg(Message).WindowPos^ Do
        Begin
             { Override size of the window. }
             If Flags And SWP_NOSIZE = 0
                Then Begin
                     ResizeDropDown ( cX, cY );
                     { Remember the last width and height passed with SWP_SIZE
                       so that they can be passed to RepositionDropDown. }
                     Last := Point ( cX, cY );
                End;

             { Override the position of the window.  If I've overrided the
               size of the drop-down during this message, use the new cX and
               cY values, otherwise use the values stored in Last. }
             If Flags And SWP_NOMOVE = 0
                Then If Flags And SWP_NOSIZE = 0
                        Then RepositionDropDown ( X, Y, cX, cY )
                        Else RepositionDropDown ( X, Y, Last.X, Last.Y );
        End;

     { Call the old window procedure to continue processing of window messages. }
     With Message Do
          Result := CallWindowProc ( TFarProc ( FOldWndProc ), FComboListHandle, Msg, wParam, lParam );
End;

Function TCustomAdjustComboBox.MaxWidth : Integer;
Var
   nItem   : Integer;
   Width   : Integer;
   IC      : HDC;
Begin
     IC := CreateIC ( 'DISPLAY', Nil, Nil, Nil );
     Canvas.Handle := IC;
     Try
        Result := 0;
        For nItem := 0 To Items.Count-1 Do
        Begin
             Width := Canvas.TextWidth ( Items[nItem] );
             If Width > Result
                Then Result := Width;
        End;
     Finally
            Canvas.Handle := 0;
            DeleteDC ( IC );
     End;
End;

Procedure TCustomAdjustComboBox.RepositionDropDown ( Var X, Y : Integer; cX, cY : Integer );
Var
   DropDownOrg,
   ParentOrg    : TPoint;
   Save         : TPoint;
Begin
     {************************ RCW 7/2/96 **************************}
     { This is just a little tricky, but not too bad.  First, I     }
     { translate all coordinates from client to screen.  Next, I    }
     { check (using the passed width and height) if the given       }
     { coordinates will cause part of the drop-down to fall off-    }
     { screen.  If so, I attempt to move the position of the drop-  }
     { down so that the full box will be visible.                   }
     {**************************************************************}

     { Translate all client coordinates to screen coordinates }
     ParentOrg   := Point ( 0, 0 );
     DropDownOrg := Point ( X, Y );

     WinProcs.ClientToScreen ( Handle, ParentOrg );
     WinProcs.ClientToScreen ( FComboListHandle, DropDownOrg );

     { Make sure the dropdown fits left<->right.  The box defaults to
       left-aligned, so I check if the box will go past the right side
       of the screen _and_ if there is enough room to shift the box to
       right-alignment.  If so, I align the right side of the box with
       the right side of the parent window.  Otherwise, I leave it alone. }
     If ( DropDownOrg.X + cX > Screen.Width ) And
        ( ParentOrg.X + Width - cX + 1 > 0 )
        Then DropDownOrg.X := ParentOrg.X + Width - cX + 1;

     { Make sure the dropdown fits top<->bottom.  Again, the default alignment
       is top, so I only change it to bottom if it will go past the bottom of
       the screen _and_ if there is enough room to place it on top of the
       parent window. }
     If ( DropDownOrg.Y + cY > Screen.Height ) And
        ( ParentOrg.Y - cY > 0 )
        Then DropDownOrg.Y := ParentOrg.Y - cY;

     { Re-translate screen coordinates to client coordinates }
     WinProcs.ScreenToClient ( FComboListHandle, DropDownOrg );

     { Return the modified X,Y coordinates }
     X := DropDownOrg.X;
     Y := DropDownOrg.Y;
End;

Procedure TCustomAdjustComboBox.ResizeDropDown ( Var cX, cY : Integer );
Var
   Cells       : Integer;

   ExtraWidth,
   ParentWidth : Integer;
Begin
     {********************* RCW 7/2/96 *****************************}
     { This is a replacement for the AdjustDropDown function        }
     { provided in the VCL.  It has been improved in a number of    }
     { places.  1)  For a csOwnerDrawFixed ComboBox, the stated     }
     {              ItemIndex is used to determine Item Height (what}
     {              a concept, I know!).  For a csOwnerDrawVariable }
     {              ComboBox, the result of a call to MeasureItem   }
     {              is used to determine Item Height.  This, IMHO,  }
     {              substantially improves upon the VCL, which      }
     {              always uses the height of the current font when }
     {              determining Item Height.                        }
     {          2)  For ComboBoxes with extra-long strings, you can }
     {              choose (through the DropDownWidth property) to  }
     {              have the DropDown size itself to fully display  }
     {              the widest string in the list.                  }
     {                                                              }
     { Things I havent' done:  I specifically haven't tried to get  }
     { the ItemHeight for each Item about to be displayed by the    }
     { ComboBox and size the DropDown accordingly.  This will only  }
     { make a difference for ComboBoxes with variable-height rows,  }
     { and if I did it in those cases, scrolling would be very      }
     { jerky.  Also, I haven't allowed a lot of leeway in the       }
     { selection of DropDownWidths.  I don't see a whole lot of use }
     { to any Width settings other than those provided.  If you do, }
     { feel free to add them.                                       }
     {**************************************************************}

     { Determine the number of visible items }
     If Items.Count = 0
        Then Cells := 1
        Else If Items.Count <= DropDownCount
             Then Cells := Items.Count
             Else Cells := DropDownCount;

     { Determine the height of the drop-down list box using ItemHeight.  If
       this is a csOwnerDrawVariable ComboBoc, give the user a chance to
       adjust the ItemHeight through a call to MeasureItem. }
     cY := ItemHeight;
     If Style = csOwnerDrawVariable
        Then MeasureItem ( 1, cY );
     cY := Cells * cY + 2; { 2: account for the border }

     { ComboBoxes with the csDropDown style display the DropDown list
       indented by 8 pixels.  I've followed this format here. }
     If Style = csDropDown
        Then ParentWidth := Width - 8
        Else ParentWidth := Width;

     { If FDropDownWidth = ddwParent, then size the list box to the parent's
       width.  Otherwise, size the ComboBox to the maximum width of the
       items in the list, or to the parent's width, whichever is greate. }
     If FDropDownWidth = ddwParent
        Then cX := ParentWidth
        Else Begin
             { Allow 10 pixels of extra space, including the vertical scroll bar,
               if it is visible. }
             If Items.Count > DropDownCount
                Then ExtraWidth := 10 + GetSystemMetrics ( SM_CXVSCROLL )
                Else ExtraWidth := 10;

             cX := MaxWidth + ExtraWidth;
             If ParentWidth > cX
                Then cX := ParentWidth;
        End;
End;




Procedure TCustomAdjustComboBox.Reset;
Begin
     RecreateWnd;
End;

end.
